#include-once
#include <WindowsConstants.au3>
#include <Array.au3>
#include <WinAPISysWin.au3>
#include <WinAPISys.au3>
#include <WinAPIConstants.au3>
#include <Misc.au3>
#include <WinAPIProc.au3>

; #INDEX# =======================================================================================================================
; Title .........: EasyHotKey UDF
; AutoIt Version : 3.3.14.5
; Language ......: English
; Author(s) .....: ISI360
; UDF Version ...: 1.1
; Description ...: With this UDF you can set various Hotkeys for your AutoIt application.
;				   The UDF uses an Application wide Keyboard Hook (WH_KEYBOARD) via wolf9228 "Global SetWindowsHookEx UDF" and
;				   calls your user defined functions via an AdlibRegister Function.
;				   This does not slow down the processing of the keyboard accelerator table chain of windows an other apps.
;				   The Hotkeys can be fine tuned via various flags or window/control/CLASS specific restrictions.
;				   You can also hook external Applications and add them to your hotkey-chain of your application.
;
;				   ADDITIONAL NOTES:
;					- This UDF and the used .dll file are based on wolf9228´s "Global SetWindowsHookEx UDF". Get it from here:
;					   https://www.autoitscript.com/forum/topic/127682-dll-global-setwindowshookex
;					- This UDF needs at least one GUI in your app! (Do not need to be visible)
;					- With wolf9228 "Global SetWindowsHookEx UDF" the hotkeys do not use the typical low level hook chain.
;					  So no "random" removal of the hotkeys. And the hotkeys in this app should not effect, or slow down other apps. (like WH_KEYBOARD_LL does)
;					- This UDF Calls Funcs in "multi threaded" way. So if you register for example two hotkeys with the same keycode,
;					  but with a different func, the funcs are called at the same time using an internal Adlibregister function.
;					  The used Adlib is unregistered automatically. 4 Funcs can be called at the same time.
; ===============================================================================================================================

; #CURRENT# =====================================================================================================================
;_EasyHotKey_Initialize
;_EasyHotKey_Shutdown
;_EasyHotKey_Pause
;_EasyHotKey_Resume
;_EasyHotKey_AssignNewHotkey
;_EasyHotKey_RemoveHotkey
;_EasyHotKey_RegisterAdditionalThread
;_EasyHotKey_RegisterFuncForDebugLogs
; ===============================================================================================================================

; #INTERNAL_USE_ONLY# ===========================================================================================================
;_EasyHotKey_AddVirtualHotkey
;_EasyHotkey_OverlappingCallAllowed
;_EasyHotkey_CallHotkeyDownFunc
;_EasyHotkey_CallHotkeyFuncAdlib_Thread1
;_EasyHotkey_CallHotkeyFuncAdlib_Thread2
;_EasyHotkey_CallHotkeyFuncAdlib_Thread3
;_EasyHotkey_CallHotkeyFuncAdlib_Thread4
;_EasyHotKey_MakeMultiThreadCall
;_EasyHotkey_KeyDownEvent
;_EasyHotkey_KeyUpEvent
;_EasyHotKey_KeyboardCallback
;_EasyHotKey_MouseCallback
;_EasyHotKey_KeyboardCallback
;_EasyHotKey_TranslateKeycodesFromIsPressed
; ===============================================================================================================================

; #EXAMPLE# =====================================================================================================================
#cs
	Example #1:
	_EasyHotKey_Initialize()
	_EasyHotKey_AssignNewHotkey("11+46","MyFunC",$EasyHotKey_AllowHoldDown,$hGUI) ;Assign ctrl + F to $hGUI

	Example #2:
	_EasyHotKey_Initialize($EasyHotKey_InitializeKeyboard) ;Use keyboard only (no mouse)
	_EasyHotKey_AssignNewHotkey("11+46","MyFunC",$EasyHotKey_AllowHotkeyForChildGUIs,$hGUI,"edit") ;Assign ctrl + F to all edit controls in $hGUI and it´s sub windows
	_EasyHotKey_RegisterFuncForDebugLogs("MyLog") ;Send all Logs of the UDF to the "MyLog" Func

#ce
; ===============================================================================================================================

; #DESCRIPTION# =================================================================================================================
#cs
	$EasyHotKey_SystemArray[i][0]   - Func to call for the assigned hotkey
	[i][1]   - Flags for the Hotkey
	[i][2]   - If key is currently pressed
	[i][3]   - Limit the hotkey to this GUI Handle
	[i][4]   - Name of the Hotkey variable
	[i][5]   - Additional virtual indexes (seperated with ,)


	## Flags for "_EasyHotKey_Initialize": ##

	$EasyHotKey_InitializeKeyboard
	When this flag is used at initialization, the UDF will listen to keyboard buttons.

	$EasyHotKey_InitializeMouse
	When this flag is used at initialization, the UDF will listen to mouse buttons.


	## Föags for "_EasyHotKey_AssignNewHotkey": ##

	$EasyHotKey_DefaultFlags
	This is the default flag value. Hotkeys can not be hold down by default.

	$EasyHotKey_AllowHoldDown
	When this flag is used, the hotkey can be hold down and is repeating, until the key is released.
	This flag has no effect on Mouse Hotkeys.

	$EasyHotKey_AllowHotkeyForChildGUIs
	When this flag is used, the hotkey is registered as a app-wide hotkey. Every child GUI in this process will react to it.
	Note: This flag can only be used, when a GUI Handle is specified at _EasyHotKey_AssignNewHotkey.

	$EasyHotKey_NoOverlappingCall
	You can use this flag to ensure, that the hotkey func call is not overlapping with another call.
	For example when $EasyHotKey_AllowHotkeyForChildGUIs is used and multiple Hotkeys are registred with the same Keycode at different GUIs.
	Should be assigned to additional hotkeys only!

	$EasyHotKey_NoHotkeyBlock
	Do not block the key for apps at lower levels in the hook chain.

#ce
; ===============================================================================================================================

; #VARIABLES# ===================================================================================================================
Global $EasyHotKey_SystemArray[0][7]
Global Const $EasyHotKey_InitializeKeyboard = 0x0002
Global Const $EasyHotKey_InitializeMouse = 0x0004
Global Const $EasyHotKey_DefaultFlags = 0x0001
Global Const $EasyHotKey_AllowHoldDown = 0x0002
Global Const $EasyHotKey_AllowHotkeyForChildGUIs = 0x0004
Global Const $EasyHotKey_NoOverlappingCall = 0x0008
Global Const $EasyHotKey_NoHotkeyBlock = 0x0010

Global $EasyHotKey_User32Dll
Global $EasyHotKey_EasyHotKeyDll
Global $EasyHotKey_CallbackHandle_Keyboard = 0
Global $EasyHotKey_CallbackHandle_Mouse = 0
Global $EasyHotKey_AssignedKeyboardCallbackFunc
Global $EasyHotKey_AssignedMouseCallbackFunc
Global $EasyHotkey_HotkeyFuncToCallAdlib
Global $EasyHotkey_UDF_Initialized = False
Global $EasyHotkey_UDF_Enabled = False
Global $EasyHotkey_CtrlKeyPressed = False
Global $EasyHotkey_AltKeyPressed = False
Global $EasyHotkey_ShiftKeyPressed = False
Global $EasyHotKey_MultiThreadCallIndex = 0
Global $EasyHotkey_HotkeyFuncToCallAdlib_Thread1
Global $EasyHotkey_HotkeyFuncToCallAdlib_Thread2
Global $EasyHotkey_HotkeyFuncToCallAdlib_Thread3
Global $EasyHotkey_HotkeyFuncToCallAdlib_Thread4
Global $EasyHotkey_FuncToRedirectDebugLogs = ""
Global $EasyHotkey_HotkeyToRelease = ""
Global $EasyHotkey_MOUSE_MSG = 0, $EasyHotkey_KEYBOARD_MSG = 0, $EasyHotkey_MSGFILTER_MSG = 0, $EasyHotkey_SHELL_MSG = 0, $EasyHotkey_HOOK_GUI_MSG = 0, $EasyHotkey_OkTestExeHwnd = 99999
Global $EasyHotkey_MouseProcID, $EasyHotkey_MouseCode, $EasyHotkey_MouseWParam, $EasyHotkey_MouseLParam
Global $EasyHotkey_KeyboardProcID, $EasyHotkey_KeyboardCode, $EasyHotkey_KeyboardWParam, $EasyHotkey_KeyboardLParam


; #FUNCTION# ====================================================================================================================
; Name ..........: _EasyHotKey_Initialize
; Description ...: Initializes the EasyHotKey UDF an registers the Low Level Hotkey Hook.
; Syntax ........: _EasyHotKey_Initialize()
; Parameters ....: $Flags          			 - [optional] What to use for the UDF (Keyboard and/or mouse). Default is both.
; 				   $KeyboardCallbackFunc     - [optional] Define a custom Callback function for keyboard stuff
; 				   $MouseCallbackFunc     - [optional] Define a custom Callback function for mouse stuff
; Return values .: Success          - 1
;                  Failure          - -1
; Author ........: ISI360
; Modified ......:
; Remarks .......: This func will auto register "_EasyHotKey_Shutdown" as an "OnAutoItExitRegister" func.
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _EasyHotKey_Initialize($Flags = BitOR($EasyHotKey_InitializeKeyboard, $EasyHotKey_InitializeMouse), $KeyboardCallbackFunc = "_EasyHotKey_KeyboardCallback", $MouseCallbackFunc = "_EasyHotKey_MouseCallback")
	If $EasyHotkey_UDF_Initialized Then Return -1 ;UDF is already initialized

	$EasyHotKey_User32Dll = DllOpen("user32.dll")
	$EasyHotKey_EasyHotKeyDll = DllOpen(@ScriptDir & "\Data\EasyHotKey.dll")

	;Register the window hook via the DLL (Keyboard and/or mouse) to the current thread
	If BitAND($Flags, $EasyHotKey_InitializeKeyboard) Then
		$EasyHotKey_CallbackHandle_Keyboard = _EasyHotKey_SetDllGlobalWindowsHookEx($WH_KEYBOARD, $KeyboardCallbackFunc, _WinAPI_GetCurrentThreadId())
		If @error Then Return -1
		$EasyHotKey_AssignedKeyboardCallbackFunc = $KeyboardCallbackFunc
	EndIf

	If BitAND($Flags, $EasyHotKey_InitializeMouse) Then
		$EasyHotKey_CallbackHandle_Mouse = _EasyHotKey_SetDllGlobalWindowsHookEx($WH_MOUSE, $MouseCallbackFunc, _WinAPI_GetCurrentThreadId())
		If @error Then Return -1
		$EasyHotKey_AssignedMouseCallbackFunc = $MouseCallbackFunc
	EndIf

	;Register exit Func
	OnAutoItExitRegister("_EasyHotKey_Shutdown")

	;Enable the UDF
	$EasyHotkey_UDF_Enabled = True
	$EasyHotkey_UDF_Initialized = True

	_EasyHotKey_Log("EasyHotKey UDF successfully initialized with '" & $Flags & "' as flags!")

	Return 1
EndFunc   ;==>_EasyHotKey_Initialize

; #FUNCTION# ====================================================================================================================
; Name ..........: _EasyHotKey_Shutdown
; Description ...: Shutdown the EasyHotKey UDF and removes the low level hook from the hook chain. All registered hotkeys will be deleted.
; Syntax ........: _EasyHotKey_Shutdown()
; Parameters ....: None
; Return values .: Success          - 1
;                  Failure          - -1
; Author ........: ISI360
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _EasyHotKey_Shutdown()
	If Not $EasyHotkey_UDF_Initialized Then Return -1 ;UDF is not initialized

	;Disable the UDF
	$EasyHotkey_UDF_Initialized = False
	$EasyHotkey_UDF_Enabled = False

	;Unhook main hook
	If ($EasyHotKey_CallbackHandle_Keyboard) Then _WinAPI_UnhookWindowsHookEx($EasyHotKey_CallbackHandle_Keyboard)
	If ($EasyHotKey_CallbackHandle_Mouse) Then _WinAPI_UnhookWindowsHookEx($EasyHotKey_CallbackHandle_Mouse)

	;Reset assigned variables
	For $Cnt = 0 To UBound($EasyHotKey_SystemArray) - 1
		If $EasyHotKey_SystemArray[$Cnt][4] <> "" Then Assign($EasyHotKey_SystemArray[$Cnt][4], "", 2)
	Next

	;Clear variables
	$EasyHotkey_FuncToRedirectDebugLogs = ""
	$EasyHotKey_CallbackHandle_Keyboard = 0
	$EasyHotKey_CallbackHandle_Mouse = 0
	$EasyHotkey_HotkeyToRelease = ""
	$EasyHotKey_AssignedMouseCallbackFunc = ""
	$EasyHotKey_AssignedKeyboardCallbackFunc = ""
	$EasyHotkey_MOUSE_MSG = 0
	$EasyHotkey_KEYBOARD_MSG = 0
	$EasyHotkey_MSGFILTER_MSG = 0
	$EasyHotkey_SHELL_MSG = 0
	$EasyHotkey_HOOK_GUI_MSG = 0
	$EasyHotkey_OkTestExeHwnd = 99999

	;Reset Array
	Local $TmpArray[0][7]
	$EasyHotKey_SystemArray = $TmpArray

	DllClose($EasyHotKey_User32Dll)
	DllClose($EasyHotKey_EasyHotKeyDll)

	_EasyHotKey_Log("EasyHotKey UDF shutdown was successfully!")

	Return 1
EndFunc   ;==>_EasyHotKey_Shutdown


; #FUNCTION# ====================================================================================================================
; Name ..........: _EasyHotKey_Reset
; Description ...: Deletes all registered Hotkeys. The hotkey hook(s) stays active.
; Syntax ........: _EasyHotKey_Reset()
; Parameters ....: None
; Return values .: Success          - 1
;                  Failure          - -1
; Author ........: ISI360
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _EasyHotKey_Reset()
	If Not $EasyHotkey_UDF_Initialized Then Return -1 ;UDF is not initialized

	;Disable UDF
	$EasyHotkey_UDF_Enabled = False

	;Reset assigned variables
	For $Cnt = 0 To UBound($EasyHotKey_SystemArray) - 1
		If $EasyHotKey_SystemArray[$Cnt][4] <> "" Then Assign($EasyHotKey_SystemArray[$Cnt][4], "", 2)
	Next

	;Reset Array
	Local $TmpArray[0][7]
	$EasyHotKey_SystemArray = $TmpArray

	;Enable UDF
	$EasyHotkey_HotkeyToRelease = ""
	$EasyHotkey_UDF_Enabled = True

	_EasyHotKey_Log("EasyHotKey UDF reset was successfully!")

	Return 1
EndFunc   ;==>_EasyHotKey_Reset


; #FUNCTION# ====================================================================================================================
; Name ..........: _EasyHotKey_Pause
; Description ...: Pauses the HotKey UDF and removes the App from the hotkey chain. When the UDF is paused, no hotkey will be recognized.
; Syntax ........: _EasyHotKey_Pause()
; Parameters ....: None
; Return values .: Success          - 1
;                  Failure          - -1
; Author ........: ISI360
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _EasyHotKey_Pause()
	If Not $EasyHotkey_UDF_Initialized Then Return -1 ;UDF is not initialized

	$EasyHotkey_UDF_Enabled = False

	;Reset all "pressed" key states
	For $Cnt = 0 To UBound($EasyHotKey_SystemArray) - 1
		$EasyHotKey_SystemArray[$Cnt][2] = 0
	Next

	_EasyHotKey_Log("EasyHotKey UDF status is set to PAUSED!")

	Return 1
EndFunc   ;==>_EasyHotKey_Pause

; #FUNCTION# ====================================================================================================================
; Name ..........: _EasyHotKey_Resume
; Description ...: Resumes the HotKey UDF and re-registerd the App in the hotkey chain.
; Syntax ........: _EasyHotKey_Resume()
; Parameters ....: None
; Return values .: Success          - 1
;                  Failure          - -1
; Author ........: ISI360
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _EasyHotKey_Resume()
	If Not $EasyHotkey_UDF_Initialized Then Return -1 ;UDF is not initialized

	$EasyHotkey_HotkeyToRelease = ""
	$EasyHotkey_UDF_Enabled = True

	_EasyHotKey_Log("EasyHotKey UDF status is set to ACTIVE!")

	Return 1
EndFunc   ;==>_EasyHotKey_Resume


; #FUNCTION# ====================================================================================================================
; Name ..........: _EasyHotKey_AssignNewHotkey
; Description ...: Registers a new Hotkey for the EasyHotKey UDF.
; Syntax ........: _EasyHotKey_AssignNewHotkey([$Keycode = "" [, $FuncToCall = "" [, $Flags = $EasyHotKey_DefaultFlags [, $LimitToGUI = "" [, $LimitToControl = ""]]]]])
; Parameters ....: $Keycode             - [optional] Keycodes for the ned hotkey. Format is from the _IsPressed command.
;													 You can combine keys with the "+"-char. Example: 11+4F
;                  $FuncToCall          - [optional] Func to call when the hotkey is pressed.
;                  $Flags               - [optional] Flags for the hotkey. Flags can be combined. See description at the beginning of the UDF.
;                  $LimitToGUI          - [optional] Limits the hotkey to a given GUI Handle at the current running process.
;													 If no handle is specified, the hotkey will be available app wide. (Same thread)
;													 Multiple Handles are possible and can be seperated with "|". Example: $hgui1&"|"&$hgui2
;                  $LimitToControl      - [optional] Limits the hotkey to a given control or a control CLASS in the GUI. (When the control has the input focus)
;													 This parameter requires a GUI via $LimitToGUI.
;													 If $LimitToControl is not set, the hotkey is assigned to the whole gui.
;													 Multiple Handles/Classes are possible and can be seperated with "|". Example: $h1&"|"&$h2
;													 A controll Handle/Class is checked via StringInStr, so you dont need to define the whole Handle/Class.
; Return values .: Success          - Index of the new assigned hotkey
;                  Failure          - -1
; Author ........: ISI360
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _EasyHotKey_AssignNewHotkey($Keycode = "", $FuncToCall = "", $Flags = $EasyHotKey_DefaultFlags, $LimitToGUI = "", $LimitToControl = "")
	If Not $EasyHotkey_UDF_Initialized Then Return -1 ;UDF is not initialized
	If $Keycode = "" Then Return -1 ;A Keycode is required!
	If $FuncToCall = "" Then Return -1 ;A Func to call is required!

	If $LimitToControl <> "" And $LimitToGUI = "" Then Return -1 ;$LimitToControl requires a GUI!

	;Setup
	Local $AdditionalKeycode = False ;True when there is already a Hotkey to this keycode, then Add a new virtual one

	;Translate IsPressed Keycodes to virtual key codes
	Local $TranslatedKeycode = _EasyHotKey_TranslateKeycodesFromIsPressed($Keycode)

	;Check if there is already a Hotkey with this keycode
	Local $MainKeyIndex = Eval("EH_" & $TranslatedKeycode)
	If String($MainKeyIndex) <> "" Then $AdditionalKeycode = True

	;Add stuff to the SystemArray
	Local $NewHotkeyIndex = _ArrayAdd($EasyHotKey_SystemArray, $FuncToCall)
	$EasyHotKey_SystemArray[$NewHotkeyIndex][1] = $Flags
	$EasyHotKey_SystemArray[$NewHotkeyIndex][2] = 0
	$EasyHotKey_SystemArray[$NewHotkeyIndex][3] = $LimitToGUI
	If Not $AdditionalKeycode Then $EasyHotKey_SystemArray[$NewHotkeyIndex][4] = "EH_" & $TranslatedKeycode
	If $AdditionalKeycode Then $EasyHotKey_SystemArray[$MainKeyIndex][5] = _EasyHotKey_AddVirtualHotkey($EasyHotKey_SystemArray[$MainKeyIndex][5], $NewHotkeyIndex)

	;Hotkey is limited to a specific control / class
	If $LimitToControl <> "" Then
		;Transform multiple control IDs to (String)-Handles
		Local $ControlLimitSplit = StringSplit($LimitToControl, "|", 2)
		If IsArray($ControlLimitSplit) Then
			$LimitToControl = ""
			For $LimitCnt = 0 To UBound($ControlLimitSplit) - 1
				If $ControlLimitSplit[$LimitCnt] = "" Then ContinueLoop
				Local $ControlWinHandle = GUICtrlGetHandle($ControlLimitSplit[$LimitCnt])
				If $ControlWinHandle <> 0 Then
					;Handle
					If $LimitToControl = "" Then $LimitToControl = "H|"
					$LimitToControl = $LimitToControl & $ControlWinHandle & "|"
				Else
					;Class
					If $LimitToControl = "" Then $LimitToControl = "C|"
					$LimitToControl = $LimitToControl & $ControlLimitSplit[$LimitCnt] & "|"
				EndIf
			Next
			If StringRight($LimitToControl, 1) = "|" Then $LimitToControl = StringTrimRight($LimitToControl, 1)
		EndIf
		$EasyHotKey_SystemArray[$NewHotkeyIndex][6] = $LimitToControl ;Set stuff in the System Array
	EndIf

	;Assign a new variable to the hotkey keycode (this can later be called directly by the Callback func)
	If Not $AdditionalKeycode Then
		If Assign("EH_" & $TranslatedKeycode, $NewHotkeyIndex, 2) = 0 Then
			_ArrayDelete($EasyHotKey_SystemArray, $NewHotkeyIndex)
			Return -1
		EndIf
	EndIf

	_EasyHotKey_Log("New Hotkey with index '" & $NewHotkeyIndex & "' and func '" & $FuncToCall & "' was registered successfully in the EasyHotKey UDF!")

	Return $NewHotkeyIndex
EndFunc   ;==>_EasyHotKey_AssignNewHotkey




; #FUNCTION# ====================================================================================================================
; Name ..........: _EasyHotKey_RemoveHotkey
; Description ...: Removes an previously registered hotkey.
; Syntax ........: _EasyHotKey_RemoveHotkey([$HotKeyIndex = -1])
; Parameters ....: $HotKeyIndex         - Hotkey index, returned from "_EasyHotKey_AssignNewHotkey"
; Return values .: None
; Author ........: ISI360
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _EasyHotKey_RemoveHotkey($HotKeyIndex = -1)
	If Not $EasyHotkey_UDF_Initialized Then Return -1 ;UDF is not initialized
	If $HotKeyIndex = -1 Then Return -1
	If $HotKeyIndex > UBound($EasyHotKey_SystemArray) - 1 Then Return -1 ;Array range exceeded

	;First, pause the UDF
	$EasyHotkey_UDF_Enabled = False

	;Setup
	Local $VariableName = $EasyHotKey_SystemArray[$HotKeyIndex][4]
	Local $AdditionalHotkeysToCheck = $EasyHotKey_SystemArray[$HotKeyIndex][5]


	;Check if hotkey is the main hotkey, or a virtual sub hotkey
	If $VariableName <> "" Then
		;Main hotkey
		Local $IndexesToCheck = StringSplit($AdditionalHotkeysToCheck, ",", 2)
		If IsArray($IndexesToCheck) Then

			;Find new main hotkey
			Local $NewMainIndex = -1
			For $Cnt = 0 To UBound($IndexesToCheck) - 1
				If $IndexesToCheck[$Cnt] = $HotKeyIndex Then ContinueLoop
				$NewMainIndex = $IndexesToCheck[$Cnt]
				_ArrayDelete($IndexesToCheck, $Cnt) ;Exclude the new main
				ExitLoop
			Next

			;And migrate the additional hotkeys and variable names to them (filtered with its own index)
			If $NewMainIndex <> -1 Then
				$EasyHotKey_SystemArray[$NewMainIndex][5] = _ArrayToString($IndexesToCheck, ",") ;migrate to new
				$EasyHotKey_SystemArray[$HotKeyIndex][5] = "" ;and remove old

				$EasyHotKey_SystemArray[$NewMainIndex][4] = $EasyHotKey_SystemArray[$HotKeyIndex][4] ;migrate to new
				$EasyHotKey_SystemArray[$HotKeyIndex][4] = "" ;and remove old

				Assign($VariableName, $NewMainIndex, 2) ;Update the control variable to the new index
			EndIf

			;Clear the array content (but do not remove the row, to ensure the returned indexes stay intact)
			$EasyHotKey_SystemArray[$HotKeyIndex][0] = ""
			$EasyHotKey_SystemArray[$HotKeyIndex][1] = ""
			$EasyHotKey_SystemArray[$HotKeyIndex][2] = ""
			$EasyHotKey_SystemArray[$HotKeyIndex][3] = ""
			$EasyHotKey_SystemArray[$HotKeyIndex][6] = ""
		EndIf

	Else
		;Virtual additional hotkey

		;First, find the main
		Local $MainIndex = -1
		For $Cnt = 0 To UBound($EasyHotKey_SystemArray) - 1
			If $Cnt = $HotKeyIndex Then ContinueLoop
			If StringInStr(String($EasyHotKey_SystemArray[$Cnt][5]), String($HotKeyIndex)) Then
				$MainIndex = $Cnt
				ExitLoop
			EndIf
		Next
		If $MainIndex = -1 Then Return -1 ;Something goes very wrong here...

		;Remove me from the main additional list
		Local $IndexesToCheck = StringSplit($EasyHotKey_SystemArray[$MainIndex][5], ",", 2)
		If IsArray($IndexesToCheck) Then
			For $Cnt = 0 To UBound($IndexesToCheck) - 1
				If $IndexesToCheck[$Cnt] = $HotKeyIndex Then
					_ArrayDelete($IndexesToCheck, $Cnt) ;Exclude me from the list
					ExitLoop
				EndIf
			Next
			$EasyHotKey_SystemArray[$MainIndex][5] = _ArrayToString($IndexesToCheck, ",") ;Create new list

			;Clear the array content (but do not remove the row, to ensure the returned indexes stay intact)
			$EasyHotKey_SystemArray[$HotKeyIndex][0] = ""
			$EasyHotKey_SystemArray[$HotKeyIndex][1] = ""
			$EasyHotKey_SystemArray[$HotKeyIndex][2] = ""
			$EasyHotKey_SystemArray[$HotKeyIndex][3] = ""
			$EasyHotKey_SystemArray[$HotKeyIndex][6] = ""
		EndIf
	EndIf

	;Re-Enable the UDF
	$EasyHotkey_UDF_Enabled = True

	_EasyHotKey_Log("Hotkey with index  " & $HotKeyIndex & " was successfully removed from the EasyHotKey UDF!")

	Return 1
EndFunc   ;==>_EasyHotKey_RemoveHotkey

; #FUNCTION# ====================================================================================================================
; Name ..........: _EasyHotKey_RegisterAdditionalThread
; Description ...: Registers an external thread (like a different process) to the hotkey udf.
;				   Callback funcs are the same, as used in "_EasyHotKey_Initialize".
;				   Warning: Registered processes may react a little bit slower, due to hooking in it´s keyboard/mouse chain.
; Syntax ........: _EasyHotKey_RegisterAdditionalThread([$ThreadID = -1])
; Parameters ....: $ThreadID            - ThreadID of the external process. (Can be find out with "_WinAPI_GetWindowThreadProcessId")
; Parameters ....: $Flags         	    - Register keyboard and/or mouse hook in the additional thread
; Return values .: Array with the new hook procedures.
;				   [0] = Handle to the new keyboard hook procedure (if used)
;				   [1] = Handle to the new mouse hook procedure (if used)
; Author ........: ISI360
; Modified ......:
; Remarks .......: NOTE: The UDF do NOT unhook the new registered thread for you!! You have to do it by your selfe with "_WinAPI_UnhookWindowsHookEx"!!
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _EasyHotKey_RegisterAdditionalThread($ThreadID = -1, $Flags = BitOR($EasyHotKey_InitializeKeyboard, $EasyHotKey_InitializeMouse))
	If Not $EasyHotkey_UDF_Initialized Then Return -1 ;UDF is already initialized
	If $ThreadID = -1 Then Return -1 ;Invalid thread ID

	Local $Result[2]

	;Register the window hook via the DLL (Keyboard and/or mouse) to the current thread
	If BitAND($Flags, $EasyHotKey_InitializeKeyboard) Then
		Local $New_Keyboard_Hook = _EasyHotKey_SetDllGlobalWindowsHookEx($WH_KEYBOARD, $EasyHotKey_AssignedKeyboardCallbackFunc, $ThreadID)
		If @error Then Return -1
		$Result[0] = $New_Keyboard_Hook
	EndIf

	If BitAND($Flags, $EasyHotKey_InitializeMouse) Then
		Local $New_Mouse_Hook = _EasyHotKey_SetDllGlobalWindowsHookEx($WH_MOUSE, $EasyHotKey_AssignedMouseCallbackFunc, $ThreadID)
		If @error Then Return -1
		$Result[1] = $New_Mouse_Hook
	EndIf

	_EasyHotKey_Log("EasyHotKey UDF registered a new thread with id '" & $ThreadID & "' to the running thread!")

	Return $Result
EndFunc   ;==>_EasyHotKey_RegisterAdditionalThread

; #FUNCTION# ====================================================================================================================
; Name ..........: _EasyHotKey_RegisterFuncForDebugLogs
; Description ...: Registers a external func, where the Hotkey UDF passes debug and log infos.
; Syntax ........: _EasyHotKey_RegisterFuncForDebugLogs([$FuncName = ""])
; Parameters ....: $FuncName            - Func to pass the debug and log infos.
; Return values .: Success          - 1
;                  Failure          - -1
; Author ........: ISI360
; Modified ......:
; Remarks .......:
; Related .......: The Debug text will be passed as first parameter to the given func.
;				   To deactivate the logging again, simply recall this func and pass "" as FuncName.
;				   NOTE: Enabled logging may slow down the UDF.
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _EasyHotKey_RegisterFuncForDebugLogs($FuncName = "")
	$EasyHotkey_FuncToRedirectDebugLogs = $FuncName
	Return 1
EndFunc   ;==>_EasyHotKey_RegisterFuncForDebugLogs


; #FUNCTION# ====================================================================================================================
; Name ..........: _EasyHotKey_AddVirtualHotkey
; Description ...:
; Syntax ........: _EasyHotKey_AddVirtualHotkey([$String = "" [, $ValueToAdd = ""]])
; Parameters ....: $String              - [optional] A string value. Default is "".
;                  $ValueToAdd          - [optional] A variant value. Default is "".
; Return values .: None
; Author ........: ISI360
; Modified ......:
; Remarks .......: Internal use only!
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _EasyHotKey_AddVirtualHotkey($String = "", $ValueToAdd = "")
	Local $NewString = $String
	$NewString = $NewString & "," & $ValueToAdd
	If StringLeft($NewString, 1) = "," Then $NewString = StringTrimLeft($NewString, 1)
	Return $NewString
EndFunc   ;==>_EasyHotKey_AddVirtualHotkey

; #FUNCTION# ====================================================================================================================
; Name ..........: _EasyHotkey_OverlappingCallAllowed
; Description ...:
; Syntax ........: _EasyHotkey_OverlappingCallAllowed(ByRef $Array [, $Cnt = 0 [, $ActiveWindow = ""]])
; Parameters ....: ByRef $Array         - An unknown value.
;                  $Cnt                 - [optional] An unknown value. Default is 0.
;                  $ActiveWindow        - [optional] An array of unknowns. Default is "".
; Return values .: None
; Author ........: ISI360
; Modified ......:
; Remarks .......: Internal use only!
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _EasyHotkey_OverlappingCallAllowed(ByRef $Array, $Cnt = 0, $ActiveWindow = "")
	Local $OverlappingCallAllowed = True
	If $Cnt > UBound($Array) - 1 Then Return $OverlappingCallAllowed
	Local $Systemindex = $Array[$Cnt]
	Local $LimitedWindow = $EasyHotKey_SystemArray[$Systemindex][3]
	For $count = $Cnt To UBound($Array) - 1
		$Systemindex = $Array[$count]
		Local $Flags = $EasyHotKey_SystemArray[$Systemindex][1]
		If BitAND($Flags, $EasyHotKey_NoOverlappingCall) And String($ActiveWindow) <> String($LimitedWindow) Then $OverlappingCallAllowed = False
	Next
	Return $OverlappingCallAllowed
EndFunc   ;==>_EasyHotkey_OverlappingCallAllowed


; #FUNCTION# ====================================================================================================================
; Name ..........: _EasyHotkey_CallHotkeyDownFunc
; Description ...:
; Syntax ........: _EasyHotkey_CallHotkeyDownFunc([$KeyIndex = 0 [, $KeyCode=""]])
; Parameters ....: $KeyIndex            - [optional] An unknown value. Default is 0.
;                  $KeyCode             - [optional] An unknown value. Default is "".
; Return values .: None
; Author ........: ISI360
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _EasyHotkey_CallHotkeyDownFunc($KeyIndex = 0, $Keycode = "")

	Local $ReturnCode = 0
	If Not $EasyHotkey_UDF_Enabled Then Return 0
	Local $MainIndex = $KeyIndex
	If $KeyIndex > UBound($EasyHotKey_SystemArray) - 1 Then Return 0 ;Array range exceeded

	Local $ActiveWindow = WinGetHandle("[active]")
	Local $ActiveControl = ControlGetHandle($ActiveWindow, "", ControlGetFocus($ActiveWindow, ""))
	Local $ActiveControlClass = ControlGetFocus($ActiveWindow)
	If StringInStr($ActiveControlClass, "edit") Then ;Get parent for combo controls
		Local $ControlParent = _WinAPI_GetParent($ActiveControl)
		If $ControlParent <> $ActiveWindow Then $ActiveControl = $ControlParent
	EndIf

	Local $AdditionalHotkeysToCheck = $EasyHotKey_SystemArray[$KeyIndex][5]
	Local $ToSplit = $KeyIndex & "," & $AdditionalHotkeysToCheck
	If StringRight($ToSplit, 1) = "," Then $ToSplit = StringTrimRight($ToSplit, 1)
	Local $IndexesToCheck = StringSplit($ToSplit, ",", 2)
	If Not IsArray($IndexesToCheck) Then Return 0

	;Loop through registered hotkeys on this keycode
	For $HotkeyCnt = 0 To UBound($IndexesToCheck) - 1
		If $IndexesToCheck[$HotkeyCnt] = "" Then ContinueLoop


		$KeyIndex = $IndexesToCheck[$HotkeyCnt]

		Local $Flags = $EasyHotKey_SystemArray[$KeyIndex][1]
		Local $FuncToCall = $EasyHotKey_SystemArray[$KeyIndex][0]
		Local $LimitedWindow = $EasyHotKey_SystemArray[$KeyIndex][3]
		Local $LimitedControl = $EasyHotKey_SystemArray[$KeyIndex][6]


		;Check if this hotkey is allowed in this gui/control
		If $LimitedWindow <> "" Then
			If BitAND($Flags, $EasyHotKey_AllowHotkeyForChildGUIs) Then
				If Not StringInStr(String($LimitedWindow), String(_WinAPI_GetParent($ActiveWindow))) And Not StringInStr(String($LimitedWindow), String(_WinAPI_GetWindow($ActiveWindow, $GW_OWNER))) And Not StringInStr(String($LimitedWindow), String(_WinAPI_GetAncestor($ActiveWindow, $GA_ROOT))) Then
					_EasyHotKey_Log("KeyDown stopped at 'GUI not allowed #1'! Func '" & $FuncToCall & "' is here not allowed!")
					ContinueLoop
				EndIf
			Else
				If Not StringInStr(String($LimitedWindow), String($ActiveWindow)) Then
					_EasyHotKey_Log("KeyDown stopped at 'GUI not allowed #2'! Func '" & $FuncToCall & "' is here not allowed!")
					ContinueLoop ;Stop here, if GUI do not match
				EndIf
			EndIf

			If Not _EasyHotkey_OverlappingCallAllowed($IndexesToCheck, $HotkeyCnt, $ActiveWindow) Then
				$ReturnCode = 1
				_EasyHotKey_Log("KeyDown stopped at 'OverlappingCallAllowed'!")
				ContinueLoop ;Stop call, if another gui in the chain uses "NoOverlappingCall"
			EndIf

		EndIf

		;Check if the hotkey is limited to a specific control
		If $LimitedControl <> "" Then
			;Control CLASS/handle
			If StringLeft($LimitedControl, 1) = "H" Then
				;Handle
				If Not StringInStr(StringTrimLeft($LimitedControl, 2), String($ActiveControl)) Then ContinueLoop
			Else
				;Class
				If Not StringInStr(StringTrimLeft($LimitedControl, 2), String($ActiveControlClass)) And Not StringInStr(String($ActiveControlClass), StringTrimLeft($LimitedControl, 2)) Then ContinueLoop
			EndIf
		EndIf

		;Calls the defined function
		If BitAND($Flags, $EasyHotKey_AllowHoldDown) Then
			;Allow key hold down and multi call
			$EasyHotKey_SystemArray[$KeyIndex][2] = 1 ;set key as pressed
			$ReturnCode = 1
			_EasyHotKey_Log("EasyHotKey UDF reacts to keypress with index " & $KeyIndex & ", keycode '" & $Keycode & "' and func '" & $FuncToCall & "' !")
			_EasyHotKey_MakeMultiThreadCall($FuncToCall)
		Else
			;Wait, until key combo is released
			If $EasyHotKey_SystemArray[$KeyIndex][2] = 0 Then ;only allow one call
				$EasyHotkey_HotkeyToRelease = $Keycode
				$EasyHotKey_SystemArray[$KeyIndex][2] = 1 ;set key as pressed
				$ReturnCode = 1
				_EasyHotKey_Log("EasyHotKey UDF reacts to keypress with index " & $KeyIndex & ", keycode '" & $Keycode & "' and func '" & $FuncToCall & "' !")
				_EasyHotKey_MakeMultiThreadCall($FuncToCall)
			Else
				$ReturnCode = 1 ;Stop, key is still pressed
			EndIf
		EndIf

	Next

	If BitAND($Flags, $EasyHotKey_NoHotkeyBlock) Then $ReturnCode = 0
	Return $ReturnCode
EndFunc   ;==>_EasyHotkey_CallHotkeyDownFunc



; #FUNCTION# ====================================================================================================================
; Name ..........: _EasyHotkey_CallHotkeyFuncAdlib_Thread1
; Description ...:
; Syntax ........: _EasyHotkey_CallHotkeyFuncAdlib_Thread1()
; Parameters ....: None
; Return values .: None
; Author ........: ISI360
; Modified ......:
; Remarks .......: Internal use only!
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _EasyHotkey_CallHotkeyFuncAdlib_Thread1()
	AdlibUnRegister("_EasyHotkey_CallHotkeyFuncAdlib_Thread1")
	Call($EasyHotkey_HotkeyFuncToCallAdlib_Thread1)
EndFunc   ;==>_EasyHotkey_CallHotkeyFuncAdlib_Thread1

; #FUNCTION# ====================================================================================================================
; Name ..........: _EasyHotkey_CallHotkeyFuncAdlib_Thread2
; Description ...:
; Syntax ........: _EasyHotkey_CallHotkeyFuncAdlib_Thread2()
; Parameters ....: None
; Return values .: None
; Author ........: ISI360
; Modified ......:
; Remarks .......: Internal use only!
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _EasyHotkey_CallHotkeyFuncAdlib_Thread2()
	AdlibUnRegister("_EasyHotkey_CallHotkeyFuncAdlib_Thread2")
	Call($EasyHotkey_HotkeyFuncToCallAdlib_Thread2)
EndFunc   ;==>_EasyHotkey_CallHotkeyFuncAdlib_Thread2

; #FUNCTION# ====================================================================================================================
; Name ..........: _EasyHotkey_CallHotkeyFuncAdlib_Thread3
; Description ...:
; Syntax ........: _EasyHotkey_CallHotkeyFuncAdlib_Thread3()
; Parameters ....: None
; Return values .: None
; Author ........: ISI360
; Modified ......:
; Remarks .......: Internal use only!
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _EasyHotkey_CallHotkeyFuncAdlib_Thread3()
	AdlibUnRegister("_EasyHotkey_CallHotkeyFuncAdlib_Thread3")
	Call($EasyHotkey_HotkeyFuncToCallAdlib_Thread3)
EndFunc   ;==>_EasyHotkey_CallHotkeyFuncAdlib_Thread3

; #FUNCTION# ====================================================================================================================
; Name ..........: _EasyHotkey_CallHotkeyFuncAdlib_Thread4
; Description ...:
; Syntax ........: _EasyHotkey_CallHotkeyFuncAdlib_Thread4()
; Parameters ....: None
; Return values .: None
; Author ........: ISI360
; Modified ......:
; Remarks .......: Internal use only!
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _EasyHotkey_CallHotkeyFuncAdlib_Thread4()
	AdlibUnRegister("_EasyHotkey_CallHotkeyFuncAdlib_Thread4")
	Call($EasyHotkey_HotkeyFuncToCallAdlib_Thread4)
EndFunc   ;==>_EasyHotkey_CallHotkeyFuncAdlib_Thread4


; #FUNCTION# ====================================================================================================================
; Name ..........: _EasyHotKey_MakeMultiThreadCall
; Description ...:
; Syntax ........: _EasyHotKey_MakeMultiThreadCall([$FuncToCall=""])
; Parameters ....: $FuncToCall          - [optional] A floating point number value. Default is "".
; Return values .: None
; Author ........: ISI360
; Modified ......:
; Remarks .......: Internal use only!
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _EasyHotKey_MakeMultiThreadCall($FuncToCall = "")
	If $EasyHotKey_MultiThreadCallIndex >= 4 Then $EasyHotKey_MultiThreadCallIndex = 0
	$EasyHotKey_MultiThreadCallIndex = $EasyHotKey_MultiThreadCallIndex + 1

	Switch $EasyHotKey_MultiThreadCallIndex

		Case 1
			$EasyHotkey_HotkeyFuncToCallAdlib_Thread1 = $FuncToCall
			AdlibRegister("_EasyHotkey_CallHotkeyFuncAdlib_Thread1", 1)


		Case 2
			$EasyHotkey_HotkeyFuncToCallAdlib_Thread2 = $FuncToCall
			AdlibRegister("_EasyHotkey_CallHotkeyFuncAdlib_Thread2", 1)

		Case 3
			$EasyHotkey_HotkeyFuncToCallAdlib_Thread3 = $FuncToCall
			AdlibRegister("_EasyHotkey_CallHotkeyFuncAdlib_Thread3", 1)

		Case 4
			$EasyHotkey_HotkeyFuncToCallAdlib_Thread4 = $FuncToCall
			AdlibRegister("_EasyHotkey_CallHotkeyFuncAdlib_Thread4", 1)

	EndSwitch
EndFunc   ;==>_EasyHotKey_MakeMultiThreadCall




; #FUNCTION# ====================================================================================================================
; Name ..........: _EasyHotkey_KeyDownEvent
; Description ...:
; Syntax ........: _EasyHotkey_KeyDownEvent([$vkCode = ""])
; Parameters ....: $vkCode              - [optional] A variant value. Default is "".
; Return values .: None
; Author ........: ISI360
; Modified ......:
; Remarks .......: Internal use only!
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _EasyHotkey_KeyDownEvent($vkCode = "")
	Local $KeyToCheck = ""
	Local $ReturnCode = 0
	If $EasyHotkey_HotkeyToRelease <> "" Then Return 1

	;Check modifier keys
	If _IsPressed("10", $EasyHotKey_User32Dll) Then
		$EasyHotkey_ShiftKeyPressed = True
	Else
		$EasyHotkey_ShiftKeyPressed = False
	EndIf

	If _IsPressed("11", $EasyHotKey_User32Dll) Then
		$EasyHotkey_CtrlKeyPressed = True
	Else
		$EasyHotkey_CtrlKeyPressed = False
	EndIf

	If _IsPressed("12", $EasyHotKey_User32Dll) Then
		$EasyHotkey_AltKeyPressed = True
	Else
		$EasyHotkey_AltKeyPressed = False
	EndIf


	Switch $vkCode

		Case 17, 162, 163 ;ctrl key
			;Ignore, the check will be done at "Check modifier keys"

		Case 18, 164, 165 ;Alt key
			;Ignore, the check will be done at "Check modifier keys"

		Case 16, 160, 161 ;Shift key
			;Ignore, the check will be done at "Check modifier keys"

		Case Else

			If $EasyHotkey_CtrlKeyPressed Then $KeyToCheck = $KeyToCheck & "17_"
			If $EasyHotkey_AltKeyPressed Then $KeyToCheck = $KeyToCheck & "18_"
			If $EasyHotkey_ShiftKeyPressed Then $KeyToCheck = $KeyToCheck & "16_"

			$KeyToCheck = $KeyToCheck & $vkCode
			;Check, if Keycode is registered
			Local $KeyIndex = Eval("EH_" & $KeyToCheck)
			If String($KeyIndex) <> "" Then
				If $KeyIndex > UBound($EasyHotKey_SystemArray) - 1 Then Return 0 ;Array range exceeded
				Local $Flags = $EasyHotKey_SystemArray[$KeyIndex][1]
				If WinGetProcess(_WinAPI_GetAncestor(WinGetHandle("[active]"), $GA_ROOT)) = @AutoItPID Or $EasyHotKey_SystemArray[$KeyIndex][3] = "" Then ;Only capture stuff from our own process.
					$ReturnCode = _EasyHotkey_CallHotkeyDownFunc($KeyIndex, $KeyToCheck) ;Check, if our app does something with this key. Return 1 to stop processing of the keyboard accelerator table (and prevent the windows "ding" sound)
				EndIf
			EndIf

	EndSwitch

	Return $ReturnCode

EndFunc   ;==>_EasyHotkey_KeyDownEvent


; #FUNCTION# ====================================================================================================================
; Name ..........: _EasyHotkey_KeyUpEvent
; Description ...:
; Syntax ........: _EasyHotkey_KeyUpEvent([$vkCode = ""])
; Parameters ....: $vkCode              - [optional] A variant value. Default is "".
; Return values .: None
; Author ........: ISI360
; Modified ......:
; Remarks .......: Internal use only!
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _EasyHotkey_KeyUpEvent($vkCode = "")

	If $EasyHotkey_HotkeyToRelease = "" Then Return 0 ;This func is only needed, when a Hotkey needs to be set as "unpressed". Hold down keys do not need this,
	Local $KeyToCheck = $EasyHotkey_HotkeyToRelease

	;Check, if Keycode is registered
	Local $KeyIndex = Eval("EH_" & $KeyToCheck)
	If String($KeyIndex) <> "" Then
		If $KeyIndex < 0 Then Return 0
		If $KeyIndex > UBound($EasyHotKey_SystemArray) - 1 Then Return 0


		Local $AdditionalHotkeysToCheck = $EasyHotKey_SystemArray[$KeyIndex][5]
		Local $ToSplit = $KeyIndex & "," & $AdditionalHotkeysToCheck
		If StringRight($ToSplit, 1) = "," Then $ToSplit = StringTrimRight($ToSplit, 1)
		Local $IndexesToCheck = StringSplit($ToSplit, ",", 2)
		If Not IsArray($IndexesToCheck) Then Return 0

		For $Cnt = 0 To UBound($IndexesToCheck) - 1
			Local $index = $IndexesToCheck[$Cnt]
			$EasyHotKey_SystemArray[$index][2] = 0 ;set key as unpressed
			$EasyHotkey_HotkeyToRelease = "" ;Reset
		Next

	EndIf
EndFunc   ;==>_EasyHotkey_KeyUpEvent




; #FUNCTION# ====================================================================================================================
; Name ..........: _EasyHotKey_KeyboardCallback
; Description ...: Default Callback func for the hotkey udf.
; Syntax ........: _EasyHotKey_KeyboardCallback($nCode, $wParam, $lParam)
; Parameters ....: $nCode               - A floating point number value.
;                  $wParam              - An unknown value.
;                  $lParam              - An unknown value.
; Return values .: None
; Author ........: ISI360
; Modified ......:
; Remarks .......: Internal use only!
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _EasyHotKey_KeyboardCallback($hWnd, $Msg, $ParamNo, $Param)

	Switch $ParamNo
		Case 1
			$EasyHotkey_KeyboardProcID = $Param
			Return 0
		Case 2
			$EasyHotkey_KeyboardCode = $Param
			Return 0
		Case 3
			$EasyHotkey_KeyboardWParam = $Param
			Return 0
		Case 4
			$EasyHotkey_KeyboardLParam = $Param
		Case Else
			Return 0
	EndSwitch

	Local $CODE = $EasyHotkey_KeyboardProcID, $wParam = $EasyHotkey_KeyboardWParam, $lParam = $EasyHotkey_KeyboardLParam, $Result
	Local $vkCode = Number($wParam)
	Local $fExtendedKey = BitAND($lParam, 2 ^ 24) <> 0
	Local $fKeyUp = BitAND($lParam, 2 ^ 31) <> 0

	If $EasyHotkey_UDF_Enabled Then
		If Not $fKeyUp Then
			$Result = _EasyHotkey_KeyDownEvent($vkCode)
		Else
			$Result = _EasyHotkey_KeyUpEvent($vkCode)
		EndIf
		If $Result <> 0 Then Return $Result
	EndIf

	Return 0
EndFunc   ;==>_EasyHotKey_KeyboardCallback




; #FUNCTION# ====================================================================================================================
; Name ..........: _EasyHotKey_MouseCallback
; Description ...: Default mouse func for the hotkey udf.
; Syntax ........: _EasyHotKey_MouseCallback($hWnd, $Msg, $ParamNo, $Param)
; Parameters ....: $hWnd                - A handle value.
;                  $Msg                 - An unknown value.
;                  $ParamNo             - A pointer value.
;                  $Param               - A pointer value.
; Return values .: None
; Author ........: ISI360
; Modified ......:
; Remarks .......:
; Related .......: Internal use only!
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _EasyHotKey_MouseCallback($hWnd, $Msg, $ParamNo, $Param)
	Switch $ParamNo
		Case 1
			$EasyHotkey_MouseProcID = $Param
			Return 0
		Case 2
			$EasyHotkey_MouseCode = $Param
			Return 0
		Case 3
			$EasyHotkey_MouseWParam = $Param
			Return 0
		Case 4
			$EasyHotkey_MouseLParam = $Param
		Case Else
			Return 0
	EndSwitch

	Local $CODE = $EasyHotkey_MouseCode, $wParam = $EasyHotkey_MouseWParam, $lParam = $EasyHotkey_MouseLParam, $Result

	If $EasyHotkey_UDF_Enabled And $CODE = 0 Then
		Switch $wParam

			Case $WM_LBUTTONDOWN
				$Result = _EasyHotkey_KeyDownEvent("1")

			Case $WM_LBUTTONUP
				$Result = _EasyHotkey_KeyUpEvent("1")

			Case $WM_RBUTTONDOWN
				$Result = _EasyHotkey_KeyDownEvent("2")

			Case $WM_RBUTTONUP
				$Result = _EasyHotkey_KeyUpEvent("2")

			Case $WM_MBUTTONDOWN
				$Result = _EasyHotkey_KeyDownEvent("4")

			Case $WM_MBUTTONUP
				$Result = _EasyHotkey_KeyUpEvent("4")

		EndSwitch
		If $Result <> 0 Then Return $Result
	EndIf

	Return 0
EndFunc   ;==>_EasyHotKey_MouseCallback

; #FUNCTION# ====================================================================================================================
; Name ..........: _EasyHotKey_TranslateKeycodesFromIsPressed
; Description ...:
; Syntax ........: _EasyHotKey_TranslateKeycodesFromIsPressed([$Keycode = ""])
; Parameters ....: $Keycode             - [optional] An unknown value. Default is "".
; Return values .: None
; Author ........: ISI360
; Modified ......:
; Remarks .......: Internal use only!
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _EasyHotKey_TranslateKeycodesFromIsPressed($Keycode = "")
	;Translate _IsPressed Hotkeys to virtual keycodes
	$Keycode = StringReplace($Keycode, "+", @LF) ;Replace + with @lf for splitting
	Local $Hotkey_Split = StringSplit($Keycode, @LF, 2)
	If Not IsArray($Hotkey_Split) Then Return
	For $x = 0 To UBound($Hotkey_Split) - 1
		Switch $Hotkey_Split[$x]

			Case "08" ;BACKSPACE key
				$Hotkey_Split[$x] = "8"


			Case "11", "A2", "A3" ;STRG
				$Hotkey_Split[$x] = "17"


			Case "10", "A0", "A1" ;SHIFT
				$Hotkey_Split[$x] = "16"


			Case "12" ;ALT
				$Hotkey_Split[$x] = "18"


			Case "5C", "5B" ;WIN Key
				$Hotkey_Split[$x] = "91"


			Case "30" ;0 key
				$Hotkey_Split[$x] = "48"

			Case "31" ;1 key
				$Hotkey_Split[$x] = "49"

			Case "32" ;2 key
				$Hotkey_Split[$x] = "50"

			Case "33" ;3 key
				$Hotkey_Split[$x] = "51"

			Case "34" ;4 key
				$Hotkey_Split[$x] = "52"

			Case "35" ;5 key
				$Hotkey_Split[$x] = "53"

			Case "36" ;6 key
				$Hotkey_Split[$x] = "54"

			Case "37" ;7 key
				$Hotkey_Split[$x] = "55"

			Case "38" ;8 key
				$Hotkey_Split[$x] = "56"

			Case "39" ;9 key
				$Hotkey_Split[$x] = "57"

			Case "60" ;keypad 0 key
				$Hotkey_Split[$x] = "96"

			Case "61" ;keypad 1 key
				$Hotkey_Split[$x] = "97"

			Case "62" ;keypad 2 key
				$Hotkey_Split[$x] = "98"

			Case "63" ;keypad 3 key
				$Hotkey_Split[$x] = "99"

			Case "64" ;keypad 4 key
				$Hotkey_Split[$x] = "100"

			Case "65" ;keypad 5 key
				$Hotkey_Split[$x] = "101"

			Case "66" ;keypad 6 key
				$Hotkey_Split[$x] = "102"

			Case "67" ;keypad 7 key
				$Hotkey_Split[$x] = "103"

			Case "68" ;keypad 8 key
				$Hotkey_Split[$x] = "104"

			Case "69" ;keypad 9 key
				$Hotkey_Split[$x] = "105"

			Case "41" ;A key
				$Hotkey_Split[$x] = "65"

			Case "42" ;B key
				$Hotkey_Split[$x] = "66"

			Case "43" ;C key
				$Hotkey_Split[$x] = "67"

			Case "44" ;D key
				$Hotkey_Split[$x] = "68"

			Case "45" ;E key
				$Hotkey_Split[$x] = "69"

			Case "46" ;F key
				$Hotkey_Split[$x] = "70"

			Case "47" ;G key
				$Hotkey_Split[$x] = "71"

			Case "48" ;H key
				$Hotkey_Split[$x] = "72"

			Case "49" ;I key
				$Hotkey_Split[$x] = "73"

			Case "4A" ;J key
				$Hotkey_Split[$x] = "74"

			Case "4B" ;K key
				$Hotkey_Split[$x] = "75"

			Case "4C" ;L key
				$Hotkey_Split[$x] = "76"

			Case "4D" ;M key
				$Hotkey_Split[$x] = "77"

			Case "4E" ;N key
				$Hotkey_Split[$x] = "78"

			Case "4F" ;O key
				$Hotkey_Split[$x] = "79"

			Case "50" ;P key
				$Hotkey_Split[$x] = "80"

			Case "51" ;Q key
				$Hotkey_Split[$x] = "81"

			Case "52" ;R key
				$Hotkey_Split[$x] = "82"

			Case "53" ;S key
				$Hotkey_Split[$x] = "83"

			Case "54" ;T key
				$Hotkey_Split[$x] = "84"

			Case "55" ;U key
				$Hotkey_Split[$x] = "85"

			Case "56" ;V key
				$Hotkey_Split[$x] = "86"

			Case "57" ;W key
				$Hotkey_Split[$x] = "87"

			Case "58" ;X key
				$Hotkey_Split[$x] = "88"

			Case "59" ;Y key
				$Hotkey_Split[$x] = "89"

			Case "5A" ;Z key
				$Hotkey_Split[$x] = "90"

			Case "09" ;TAB key
				$Hotkey_Split[$x] = "9"

			Case "0C" ;CLEAR key
				$Hotkey_Split[$x] = "46"

			Case "0D" ;Enter key
				$Hotkey_Split[$x] = "13"

			Case "13" ;PAUSE key
				$Hotkey_Split[$x] = "19"

			Case "14" ;CAPS LOCK key
				$Hotkey_Split[$x] = "20"

			Case "1B" ;ESC key
				$Hotkey_Split[$x] = "27"

			Case "20" ;SPACEBAR
				$Hotkey_Split[$x] = "32"

			Case "21" ;PAGE UP key
				$Hotkey_Split[$x] = "33"

			Case "22" ;PAGE DOWN key
				$Hotkey_Split[$x] = "34"

			Case "23" ;END key
				$Hotkey_Split[$x] = "35"

			Case "24" ;HOME key
				$Hotkey_Split[$x] = "36"

			Case "25" ;LEFT ARROW key
				$Hotkey_Split[$x] = "37"

			Case "26" ;UP ARROW key
				$Hotkey_Split[$x] = "38"

			Case "27" ;RIGHT ARROW key
				$Hotkey_Split[$x] = "39"

			Case "28" ;DOWN ARROW key
				$Hotkey_Split[$x] = "40"

			Case "2B" ;EXECUTE
				$Hotkey_Split[$x] = "43"

			Case "5D", "A4", "A5" ;PopUp Menu Key
				$Hotkey_Split[$x] = "93"

			Case "2D" ;INS key
				$Hotkey_Split[$x] = "45"

			Case "2E" ;DEL key
				$Hotkey_Split[$x] = "46"

			Case "6A" ;Multiply key
				$Hotkey_Split[$x] = "106"

			Case "6B" ;Add key
				$Hotkey_Split[$x] = "107"

			Case "6D" ;Subtract key
				$Hotkey_Split[$x] = "109"

			Case "6E" ;Decimal key
				$Hotkey_Split[$x] = "110"

			Case "6F" ;Divide key
				$Hotkey_Split[$x] = "111"

			Case "70" ;F1 key
				$Hotkey_Split[$x] = "112"

			Case "71" ;F2 key
				$Hotkey_Split[$x] = "113"

			Case "72" ;F3 key
				$Hotkey_Split[$x] = "114"

			Case "73" ;F4 key
				$Hotkey_Split[$x] = "115"

			Case "74" ;F5 key
				$Hotkey_Split[$x] = "116"

			Case "75" ;F6 key
				$Hotkey_Split[$x] = "117"

			Case "76" ;F7 key
				$Hotkey_Split[$x] = "118"

			Case "77" ;F8 key
				$Hotkey_Split[$x] = "119"

			Case "78" ;F9 key
				$Hotkey_Split[$x] = "120"

			Case "79" ;F10 key
				$Hotkey_Split[$x] = "121"

			Case "7A" ;F11 key
				$Hotkey_Split[$x] = "122"

			Case "7B" ;F12 key
				$Hotkey_Split[$x] = "123"

			Case "DC" ; ^ key
				$Hotkey_Split[$x] = "220"

			Case "01" ; left mouse key
				$Hotkey_Split[$x] = "1"

			Case "02" ; right mouse key
				$Hotkey_Split[$x] = "2"

			Case "04" ; middle mouse key
				$Hotkey_Split[$x] = "4"

			Case Else
				$Hotkey_Split[$x] = ""

		EndSwitch
	Next

	;Convert the array to a String and sort them (modifier keys must be at first)
	Local $ModKeys = ""
	Local $NewKeyString = ""
	For $Cnt = 0 To UBound($Hotkey_Split) - 1
		If $Hotkey_Split[$Cnt] = "91" Or $Hotkey_Split[$Cnt] = "160" Or $Hotkey_Split[$Cnt] = "162" Or $Hotkey_Split[$Cnt] = "164" Then
			$ModKeys = $ModKeys & $Hotkey_Split[$Cnt] & "_"
		Else
			$NewKeyString = $NewKeyString & $Hotkey_Split[$Cnt] & "_"
		EndIf


	Next
	$NewKeyString = $ModKeys & $NewKeyString
	If StringRight($NewKeyString, 1) = "_" Then $NewKeyString = StringTrimRight($NewKeyString, 1)
	Return $NewKeyString

EndFunc   ;==>_EasyHotKey_TranslateKeycodesFromIsPressed

; #FUNCTION# ====================================================================================================================
; Name ..........: _EasyHotKey_Log
; Description ...: Used for passing logs to the registered logging function
; Syntax ........: _EasyHotKey_Log([$Text = ""])
; Parameters ....: $Text                - [optional] A dll struct value. Default is "".
; Return values .: Success          - 1
;                  Failure          - -1
; Author ........: ISI360
; Modified ......: Internal use only!
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _EasyHotKey_Log($Text = "")
	If Not $EasyHotkey_UDF_Initialized Then Return -1 ;UDF is not initialized
	If $EasyHotkey_FuncToRedirectDebugLogs = "" Then Return -1 ;No logging func is registered

	Call($EasyHotkey_FuncToRedirectDebugLogs, $Text)

	Return 1
EndFunc   ;==>_EasyHotKey_Log

; #FUNCTION# ====================================================================================================================
; Name ..........: _EasyHotKey_TestExeHwnd
; Description ...:
; Syntax ........: _EasyHotKey_TestExeHwnd($hWnd, $Msg, $wParam, $lParam)
; Parameters ....: $hWnd                - A handle value.
;                  $Msg                 - An unknown value.
;                  $wParam              - An unknown value.
;                  $lParam              - An unknown value.
; Return values .: None
; Author ........: wolf9228
; Modified ......: ISI360
; Remarks .......: Internal use only!
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _EasyHotKey_TestExeHwnd($hWnd, $Msg, $wParam, $lParam)
	Return $EasyHotkey_OkTestExeHwnd
EndFunc   ;==>_EasyHotKey_TestExeHwnd

; #FUNCTION# ====================================================================================================================
; Name ..........: _EasyHotKey_RegisterWindowMessage
; Description ...:
; Syntax ........: _EasyHotKey_RegisterWindowMessage($lpString)
; Parameters ....: $lpString            - An unknown value.
; Return values .: None
; Author ........: wolf9228
; Modified ......: ISI360
; Remarks .......: Internal use only!
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _EasyHotKey_RegisterWindowMessage($lpString)
	$RT = DllCall("User32.dll", "int", "RegisterWindowMessageW", "WSTR", $lpString)
	If @error Then Return SetError(1, 0, 0)
	Return SetError(_WinAPI_GetLastError(), 0, $RT[0])
EndFunc   ;==>_EasyHotKey_RegisterWindowMessage

; #FUNCTION# ====================================================================================================================
; Name ..........: _EasyHotKey_SetDllGlobalWindowsHookEx
; Description ...: Used to register a new thread for WH_* an the EasyHotkey.dll
; Syntax ........: _EasyHotKey_SetDllGlobalWindowsHookEx($IdHook, $MsgFunction, $ThreadID)
; Parameters ....: $IdHook              - Hook to use
;                  $MsgFunction         - Callback Func
;                  $ThreadID            - Thread ID
; Return values .: None
; Author ........: wolf9228
; Modified ......: ISI360
; Remarks .......: Internal use only!
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _EasyHotKey_SetDllGlobalWindowsHookEx($IdHook, $MsgFunction, $ThreadID)
	If Not $EasyHotKey_EasyHotKeyDll Then Return SetError(1, 0, 0)

	If Not ($EasyHotkey_HOOK_GUI_MSG) Then
		Local $RT = DllCall($EasyHotKey_EasyHotKeyDll, "BOOL", "DllGetModuleFileNameW", "WSTR*", "")
		If @error Or Not $RT[0] Then Return SetError(0, 0, 0)
		Local $MsgBuffer = $RT[1]
		$EasyHotkey_HOOK_GUI_MSG = _EasyHotKey_RegisterWindowMessage($MsgBuffer)
		If Not $EasyHotkey_HOOK_GUI_MSG Or Not GUIRegisterMsg($EasyHotkey_HOOK_GUI_MSG, "_EasyHotKey_TestExeHwnd") Then Return SetError(2, 0, 0)
	EndIf

	Switch $IdHook

		Case $WH_KEYBOARD
			If Not ($EasyHotkey_KEYBOARD_MSG) Then
				$EasyHotkey_KEYBOARD_MSG = _EasyHotKey_RegisterWindowMessage("KEYBOARD_MSG")
				If Not $EasyHotkey_KEYBOARD_MSG Or Not GUIRegisterMsg($EasyHotkey_KEYBOARD_MSG, $MsgFunction) Then Return SetError(3, 0, 0)
			EndIf

		Case $WH_MOUSE
			If Not ($EasyHotkey_MOUSE_MSG) Then
				$EasyHotkey_MOUSE_MSG = _EasyHotKey_RegisterWindowMessage("MOUSE_MSG")
				If Not $EasyHotkey_MOUSE_MSG Or Not GUIRegisterMsg($EasyHotkey_MOUSE_MSG, $MsgFunction) Then Return SetError(3, 0, 0)
			EndIf

		Case Else
			Return SetError(4, 0, 0)

	EndSwitch

	Local $RT = DllCall($EasyHotKey_EasyHotKeyDll, "handle", "DllWindowsHookExW", "UINT", $IdHook, "UINT", $ThreadID)
	If @error Or Not $RT[0] Then Return SetError(5, 0, 0)

	Switch $IdHook

		Case $WH_KEYBOARD
			$HookHandleKeyboardProc = $RT[0]

		Case $WH_MOUSE
			$HookHandleMouseProc = $RT[0]

	EndSwitch

	Return SetError(0, 0, $RT[0])
EndFunc   ;==>_EasyHotKey_SetDllGlobalWindowsHookEx
